/***************************************************************************
 *   Copyright (C) 2008-2024 by Ilya Kotov                                 *
 *   forkotov02@ya.ru                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#include <QFileInfo>
#include <QtPlugin>
#include <QRegularExpression>
#include <qmmpui/metadataformatter.h>
#include "m3uplaylistformat.h"

PlayListFormatProperties M3UPlaylistFormat::properties() const
{
    PlayListFormatProperties p;
    p.filters = QStringList { u"*.m3u"_s, u"*.m3u8"_s };
    p.shortName = "m3u"_L1;
    p.contentTypes = QStringList { u"audio/x-mpegurl"_s };
    return p;
}

QList<PlayListTrack *> M3UPlaylistFormat::decode(const QByteArray &contents)
{
    QList<PlayListTrack*> out;
    const QStringList splitted = QString::fromUtf8(contents).split(QChar::LineFeed);
    if(splitted.isEmpty())
        return out;

    static const QRegularExpression extInfRegExp(u"#EXTINF:(-{0,1}\\d+),(.*) - (.*)"_s);
    static const QRegularExpression extInfRegExp2(u"#EXTINF:(-{0,1}\\d+),(.*)"_s);
    int length = 0;
    QString artist, title;
    bool hasExtInf = false;

    for(const QString &line : std::as_const(splitted))
    {
        QString str = line.trimmed();
        if(str.startsWith(u"#EXTM3U"_s) || str.isEmpty())
            continue;

        QRegularExpressionMatch match;

        if((match = extInfRegExp.match(str)).hasMatch())
        {
            length = match.captured(1).toInt();
            artist = match.captured(2);
            title =  match.captured(3);
            hasExtInf = true;
        }
        else if((match = extInfRegExp2.match(str)).hasMatch())
        {
            length = match.captured(1).toInt();
            title =  match.captured(2);
            hasExtInf = true;
        }

        if(str.startsWith(QLatin1Char('#')))
            continue;

        out << new PlayListTrack();
        out.last()->setPath(str);

        if(hasExtInf)
        {
            out.last()->setDuration(length * 1000);
            out.last()->setValue(Qmmp::ARTIST, artist);
            out.last()->setValue(Qmmp::TITLE, title);
            hasExtInf = false;
        }
    }
    return out;
}

QByteArray M3UPlaylistFormat::encode(const QList<PlayListTrack*> &contents, const QString &path)
{
    QStringList out = { u"#EXTM3U"_s };
    MetaDataFormatter formatter(u"%if(%p,%p - %t,%t)%if(%p|%t,,%f)"_s);
    QString m3uDir = QFileInfo(path).canonicalPath();

    for(const PlayListTrack *f : std::as_const(contents))
    {
        QString info = u"#EXTINF:"_s + QString::number(f->duration() / 1000) + QLatin1Char(',') + formatter.format(f);
        out.append(info);

        if(!f->path().contains(u"://"_s) && f->path().startsWith(m3uDir))
        {
            QString p = f->path();
            p.remove(0, m3uDir.size());
            if(p.startsWith(QLatin1Char('/')))
                p.remove(0, 1);
            out.append(p);
        }
        else
            out.append(f->path());
    }
    return out.join(QChar::LineFeed).toUtf8();
}
